/******************************************************************************
 * evtchn.h
 * 
 * Communication via Xen event channels.
 * Also definitions for the device that demuxes notifications to userspace.
 * 
 * Copyright (c) 2004-2005, K A Fraser
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation; or, when distributed
 * separately from the Linux kernel or incorporated into other
 * software packages, subject to the following license:
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this source file (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef __ASM_EVTCHN_H__
#define __ASM_EVTCHN_H__

#include <linux/interrupt.h>
#include <asm/ptrace.h>
#include <linux/smp.h>

#include <xen/hypervisor.h>
#include <xen/interface/event_channel.h>

/* Internal flags */
#define IRQ_INPROGRESS		0x00000100	/* IRQ handler active - do not enter! */
#define IRQ_DISABLED		0x00000200	/* IRQ disabled - do not enter! */
#define IRQ_PENDING		0x00000400	/* IRQ pending - replay on enable */
#define IRQ_REPLAY		0x00000800	/* IRQ has been replayed but not acked yet */
#define IRQ_AUTODETECT		0x00001000	/* IRQ is being autodetected */
#define IRQ_WAITING		0x00002000	/* IRQ not yet seen - for autodetection */
#define IRQ_LEVEL		0x00004000	/* IRQ level triggered */
#define IRQ_MASKED		0x00008000	/* IRQ masked - shouldn't be seen again */
#define IRQ_PER_CPU		0x00010000	/* IRQ is per CPU */
#define IRQ_NOPROBE		0x00020000	/* IRQ is not valid for probing */
#define IRQ_NOREQUEST		0x00040000	/* IRQ cannot be requested */
#define IRQ_NOAUTOEN		0x00080000	/* IRQ will not be enabled on request irq */
#define IRQ_WAKEUP		0x00100000	/* IRQ triggers system wakeup */
#define IRQ_MOVE_PENDING	0x00200000	/* need to re-target IRQ destination */
#define IRQ_NO_BALANCING	0x00400000	/* IRQ is excluded from balancing */

struct hw_interrupt_type {
	const char 	*typename;
	unsigned int 	(*startup)(unsigned int irq);
	void 		(*shutdown)(unsigned int irq);
	void 		(*enable)(unsigned int irq);
	void 		(*disable)(unsigned int irq);
	void 		(*ack)(unsigned int irq);
	void 		(*end)(unsigned int irq);
	void 		(*set_affinity)(unsigned int irq, cpumask_t dest);
	int			(*set_type)(unsigned int irq, unsigned int type);
	void 		(*release)(unsigned int irq, void *dev_id);
};

typedef struct hw_interrupt_type  hw_irq_controller;

typedef struct evtchn_desc {
	hw_irq_controller 	*handler;
	void 			*handler_data;
	struct irqaction 	*action;		/* IRQ action list */
	unsigned int 		status;			/* IRQ status */
	unsigned int 		depth;			/* nested irq disables */
	unsigned int 		irq_count;		/* For detecting broken interrupts */
	unsigned int 		irqs_unhandled;
	spinlock_t 		lock;
#if defined (CONFIG_GENERIC_PENDING_IRQ) || defined (CONFIG_IRQBALANCE)
	unsigned int 		move_irq;		/* Flag need to re-target intr dest*/
#endif
} evtchn_desc_t;


/* Binding types. */
enum {
        IRQT_UNBOUND,
        IRQT_PIRQ,
        IRQT_VIRQ,
        IRQT_IPI,
        IRQT_EVTCHN
};

static inline void transaction_clear_bit(int port, volatile unsigned long *p)
{
	unsigned long flags;
	unsigned long tmp;
	unsigned long b,old_b;
	unsigned long mask = ~(1UL << (port & 31));
	unsigned long x,ret,mask2;

	p += port >> 5;
	
	local_irq_save(flags);
	__asm__ __volatile__("@transaction_clear_bit\n"
"	ldr	%5, [%7] \n"
"	and	%1, %5, %6 \n"
"50:	mov %4, %1\n"
"	swp	%0, %4, [%7]\n"
"	cmp	%0, %5\n"
"	eorne	%2,%0, %5\n"
"	mvnne	%3, %2\n"
"	andne	%1,%1,%3\n"
"	andne	%3, %0, %2\n"
"	orrne	%1,%1, %3\n"
"	movne %5, %4\n"
"	bne		50b\n"
	: "=&r" (x), "=&r" (ret), "=&r" (mask2),"=&r" (tmp) , "=&r" (b),"=&r" (old_b)
	:"r" (mask),  "r" (p) 
	: "cc", "memory");

	local_irq_restore(flags);
	
}

static inline void transaction_set_bit(int port, volatile unsigned long *p)
{
	unsigned long flags;
	unsigned long tmp;
	unsigned long b,old_b;
	unsigned long mask = 1UL << (port & 31);
	unsigned long x,ret,mask2;

	p += port >> 5;
	
	local_irq_save(flags);
	__asm__ __volatile__("@transaction_set_bit\n"
"	ldr	%5, [%7] \n"
"	orr	%1, %5, %6 \n"
"50:	mov	%4, %1\n"
"	swp	%0, %4, [%7]\n"
"	cmp	%0, %5\n"
"	eorne	%2,%0, %5\n"
"	mvnne	%3, %2\n"
"	andne	%1,%1,%3\n"
"	andne	%3, %0, %2\n"
"	orrne	%1,%1, %3\n"
"	movne %5, %4\n"
"	bne		50b\n"
	: "=&r" (x), "=&r" (ret), "=&r" (mask2),"=&r" (tmp) , "=&r" (b),"=&r" (old_b)
	:"r" (mask),  "r" (p) 
	: "cc", "memory");

	local_irq_restore(flags);
}

/* Inline Functions */
/* Constructor for packed IRQ information. */
static inline u32 mk_irq_info(u32 type, u32 index, u32 evtchn) 
{
        return ((type << 24) | (index << 16) | evtchn);
}

static inline void clear_evtchn(int port)
{
	volatile shared_info_t *s = HYPERVISOR_shared_info;
	//clear_bit(port, &s->evtchn_pending[0]);
	transaction_clear_bit(port, &s->evtchn_pending[0]);
}

static inline void notify_remote_via_evtchn(int port)
{
	evtchn_op_t op;
	op.cmd         = EVTCHNOP_send,
	op.u.send.port = port;
	(void)HYPERVISOR_event_channel_op(&op);
}


/* Entry point for notifications into Linux subsystems. */
asmlinkage void evtchn_do_upcall(struct pt_regs *regs);

/*
 * LOW-LEVEL DEFINITIONS
 */

/*
 * Dynamically bind an event source to an IRQ-like callback handler.
 * On some platforms this may not be implemented via the Linux IRQ subsystem.
 * The IRQ argument passed to the callback handler is the same as returned
 * from the bind call. It may not correspond to a Linux IRQ number.
 * Returns IRQ or negative errno.
 * UNBIND: Takes IRQ to unbind from; automatically closes the event channel.
 */
extern int bind_evtchn_to_irqhandler(
		unsigned int evtchn,
		irq_handler_t handler,
		unsigned long irqflags,
		const char *devname,
		void *dev_id);

extern int bind_virq_to_irqhandler(
		unsigned int virq,
		unsigned int cpu,
		irq_handler_t handler,
		unsigned long irqflags,
		const char *devname,
		void *dev_id);

extern int bind_ipi_to_irqhandler(
		unsigned int ipi,
		unsigned int cpu,
		irq_handler_t handler,
		unsigned long irqflags,
		const char *devname,
		void *dev_id);

/*
 * Common unbind function for all event sources. Takes IRQ to unbind from.
 * Automatically closes the underlying event channel (even for bindings
 * made with bind_evtchn_to_irqhandler()).
 */
extern void unbind_from_irqhandler(unsigned int irq, void *dev_id);

extern void irq_resume(void);

/* Entry point for notifications into the userland character device. */
extern void evtchn_device_upcall(int port);

extern void mask_evtchn(int port);
extern void unmask_evtchn(int port);

/*
 * Unlike notify_remote_via_evtchn(), this is safe to use across
 * save/restore. Notifications on a broken connection are silently dropped.
 */
extern void notify_remote_via_irq(int irq);

#endif /* __ASM_EVTCHN_H__ */

/*
 * Local variables:
 *  c-file-style: "linux"
 *  indent-tabs-mode: t
 *  c-indent-level: 8
 *  c-basic-offset: 8
 *  tab-width: 8
 * End:
 */
