/******************************************************************************
 * hypervisor.h
 * 
 * Linux-specific hypervisor handling.
 * 
 * Copyright (c) 2002-2004, K A Fraser
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation; or, when distributed
 * separately from the Linux kernel or incorporated into other
 * software packages, subject to the following license:
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this source file (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef _HYPERCALL_H_
#define _HYPERCALL_H_

#include <xen/interface/xen.h>
#include <xen/interface/dom0_ops.h>
#include <xen/interface/security/sra_ops.h>
#include <xen/interface/acm_ops.h>
#include <xen/interface/sched.h>

#include <asm-generic/errno.h>


#define HYPERCALL_INSTR 	                "swi 0x82"


#define DECLARE_REG(name, reg)			register long name __asm(reg)
#define ASSIGN_REG(name, reg, value)		DECLARE_REG(name, reg) = value;

#define __SYS_REG_LIST(regs...) regs

#define xen_init()	(0)

extern start_info_t *xen_start_info;
extern volatile shared_info_t *HYPERVISOR_shared_info;

extern void		xen_flush_kern_tlb_range(unsigned long start, unsigned long end);
extern void    	xen_flush_kern_cache_all(void);
extern void    	xen_flush_user_cache_all(void);
extern void 	xen_flush_cache(void);
extern void 	xen_dcache_clean_area(void *p, int size);
extern void 	xen_flush_tlb_page(unsigned int, unsigned long uaddr);
extern void 	xen_flush_tlb_kernel_page(unsigned long kaddr);
extern void 	xen_flush_pmd_entry(pmd_t *pmd);
extern void 	xen_clean_pmd_entry(pmd_t *pmd);
extern void		xen_flush_tlb_all(void);
extern void 	xen_flush_tlb_mm(unsigned int);

void	force_evtchn_callback(void);
int 	xen_create_contiguous_region(unsigned long vstart, unsigned int order, unsigned int address_bits);
void 	xen_destroy_contiguous_region(unsigned long vstart, unsigned int order);

shared_info_t *xen_map_shared_info(unsigned long pa);


static inline int HYPERVISOR_set_trap_table(trap_info_t *table)
{
        register long a0 __asm__("r0") = (long)(table);
        register long r0 __asm__("r0");

        register long f0 __asm__("r7") = __HYPERVISOR_set_trap_table;

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_mmu_update(mmu_update_t *req, int count, int *success_count, domid_t domid)
{
        register long a0 __asm__("r0") = (long)(req);
        register long a1 __asm__("r1") = (long)(count);
        register long a2 __asm__("r2") = (long)(success_count);
        register long a3 __asm__("r3") = (long)domid;

        register long f0 __asm__("r7") = __HYPERVISOR_mmu_update;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(a2), "r"(a3), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_mmuext_op(struct mmuext_op *op, int count, int *success_count, domid_t domid)
{
        register long a0 __asm__("r0") = (long)op;
        register long a1 __asm__("r1") = (long)count;
        register long a2 __asm__("r2") = (long)success_count;
        register long a3 __asm__("r3") = (long)domid;

        register long f0 __asm__("r7") = __HYPERVISOR_mmuext_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(a2), "r"(a3), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_stack_switch(unsigned long ss, unsigned long esp)
{
        register long a0 __asm__("r0") = (long)ss;
        register long a1 __asm__("r1") = (long)esp;

        register long f0 __asm__("r7") = __HYPERVISOR_stack_switch;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_set_callbacks
(
        unsigned long event_address,
        unsigned long failsafe_address
)
{
        register long a0 __asm__("r0") = (long)event_address;
        register long a1 __asm__("r1") = (long)failsafe_address;

        register long f0 __asm__("r7") = __HYPERVISOR_set_callbacks;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_do_print_profile(int section_id)
{
        register long a0 __asm__("r0") = (long)section_id;

        register long f0 __asm__("r7") = __HYPERVISOR_do_print_profile;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_do_set_foreground_domain(unsigned int domain_id)
{
        register long a0 __asm__("r0") = (long)domain_id;

        register long f0 __asm__("r7") = __HYPERVISOR_do_set_foreground_domain;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_do_set_HID_irq(unsigned int irq)
{
        register long a0 __asm__("r0") = (long)irq;

        register long f0 __asm__("r7") = __HYPERVISOR_do_set_HID_irq;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_fpu_taskswitch(int set)
{
        register long a0 __asm__("r0") = (long)set;

        register long f0 __asm__("r7") = __HYPERVISOR_fpu_taskswitch;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_sched_op(int cmd, unsigned long arg)
{
        register long a0 __asm__("r0") = (long)cmd;
        register long a1 __asm__("r1") = (long)arg;

        register long f0 __asm__("r7") = __HYPERVISOR_sched_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline void HYPERVISOR_do_dummy(void)
{
        register long f0 __asm__("r7") = __HYPERVISOR_dummy;
		
        __asm__ __volatile__(
                HYPERCALL_INSTR
                :
                : "r" (f0)
				: "memory");
}

static inline long HYPERVISOR_set_timer_op(unsigned long timeout)
{
        register long a0 __asm__("r0") = (unsigned long)(timeout);
        register long f0 __asm__("r7") = __HYPERVISOR_set_timer_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");
        return (int)r0;
}

static inline long HYPERVISOR_get_system_time(void)
{
        register long f0 __asm__("r7") = __HYPERVISOR_get_system_time;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "r"(f0)
				: "memory");
        return (int)r0;
}


static inline int HYPERVISOR_dom0_op(dom0_op_t *dom0_op)
{
        register long a0 __asm__("r0") = (unsigned long)(dom0_op);

        register long f0 __asm__("r7") = __HYPERVISOR_dom0_op;
        register long r0 __asm__("r0");

        dom0_op->interface_version = DOM0_INTERFACE_VERSION;

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_acm_op(acm_op_t *acm_op)
{
        register long a0 __asm__("r0") = (unsigned long)(acm_op);

        register long f0 __asm__("r7") = __HYPERVISOR_acm_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_sra_op(sra_op_t *sra_op)
{
        register long a0 __asm__("r0") = (unsigned long)(sra_op);

        register long f0 __asm__("r7") = __HYPERVISOR_sra_op;
        register long r0 __asm__("r0");

        sra_op->interface_version = SRA_INTERFACE_VERSION;

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}


static inline int HYPERVISOR_memory_op(unsigned int cmd, void *arg)
{
        register long a0 __asm__("r0") = (long)cmd;
        register long a1 __asm__("r1") = (long)arg;

        register long f0 __asm__("r7") = __HYPERVISOR_memory_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_multicall(void *call_list, int nr_calls)
{
        register long a0 __asm__("r0") = (long)call_list;
        register long a1 __asm__("r1") = (long)nr_calls;

        register long f0 __asm__("r7") = __HYPERVISOR_multicall;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_update_va_mapping(unsigned long va, unsigned long new_val, unsigned long flags)
{
        register long a0 __asm__("r0") = (long)va;
        register long a3 __asm__("r1") = (long)flags;
        register long a1 __asm__("r2") = (long)new_val;
        register long a2 __asm__("r3") = 0;

        register long f0 __asm__("r7") = __HYPERVISOR_update_va_mapping;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(a2), "r"(a3), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_event_channel_op(void *op)
{
        register long a0 __asm__("r0") = (long)op;

        register long f0 __asm__("r7") = __HYPERVISOR_event_channel_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_xen_version(int cmd, void *arg)
{
        register long a0 __asm__("r0") = (long)cmd;
        register long a1 __asm__("r1") = (long)arg;

        register long f0 __asm__("r7") = __HYPERVISOR_xen_version;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_console_io(int cmd, int count, char *str)
{
	register long a0 __asm__("r0") = cmd;
	register long a1 __asm__("r1") = count;
	register long a2 __asm__("r2") = (long)str;

	register long f0 __asm__("r7") = __HYPERVISOR_console_io;
	register long r0 __asm__("r0");
	
	__asm__ __volatile__(
		HYPERCALL_INSTR
		: "=r"(r0)
		: "0"(a0), "r"(a1), "r"(a2), "r"(f0)
		: "memory");
	
	return (int)r0;
}

static inline int HYPERVISOR_physdev_op(void *physdev_op)
{
        register long a0 __asm__("r0") = (long)physdev_op;

        register long f0 __asm__("r7") = __HYPERVISOR_physdev_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_grant_table_op(unsigned int cmd, void *uop, unsigned int count)
{
        register long a0 __asm__("r0") = cmd;
        register long a1 __asm__("r1") = (long)uop;
        register long a2 __asm__("r2") = count;

        register long f0 __asm__("r7") = __HYPERVISOR_grant_table_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(a2), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_update_va_mapping_otherdomain
(
        unsigned long va,
        unsigned long new_val,
        unsigned long flags,
        domid_t domid
)
{
        register long a0 __asm__("r0") =(long)va;
        register long a1 __asm__("r1") = (long)new_val;
        register long a2 __asm__("r2") = 0;
        register long a3 __asm__("r3") = (long)flags;
        register long a4 __asm__("r4") = (long)domid;

        register long f0 __asm__("r7") = __HYPERVISOR_vcpu_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(a2), "r"(a3), "r"(a4), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_vm_assist(unsigned int cmd, unsigned int type)
{
        register long a0 __asm__("r0") = (long)cmd;
        register long a1 __asm__("r1") = (long)type;

        register long f0 __asm__("r7") = __HYPERVISOR_vm_assist;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_vcpu_op(int cmd, int vcpuid, void *extra_args)
{
        register long a0 __asm__("r0") = cmd;
        register long a1 __asm__("r1") = vcpuid;
        register long a2 __asm__("r2") = (long)extra_args;

        register long f0 __asm__("r7") = __HYPERVISOR_vcpu_op;
        register long r0 __asm__("r0");

        __asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0)
                : "0"(a0), "r"(a1), "r"(a2), "r"(f0)
				: "memory");

        return (int)r0;
}

static inline int HYPERVISOR_suspend(unsigned long srec)
{
        return 0;
}


static inline int HYPERVISOR_set_cpu_domain(unsigned long dom)
{
        register long a0 __asm__("r0") = dom;
        register long f0 __asm__("r7") = __HYPERVISOR_set_cpu_domain;
        register long r0 __asm__("r0");
		
		__asm__ __volatile__(
                HYPERCALL_INSTR
                : "=r"(r0) 
                : "0"(a0), "r"(f0)
				: "memory");

        return (int)r0;
}

/*
static inline void
MULTI_update_va_mapping(
    multicall_entry_t *mcl, unsigned long va,
    pte_t new_val, unsigned long flags)
{
    mcl->op = __HYPERVISOR_update_va_mapping;
    mcl->args[0] = va;
    mcl->args[1] = new_val;
    mcl->args[2] = 0;
    mcl->args[3] = flags;
}

static inline void
MULTI_update_va_mapping_otherdomain(
    multicall_entry_t *mcl, unsigned long va,
    pte_t new_val, unsigned long flags, domid_t domid)
{
    mcl->op = __HYPERVISOR_update_va_mapping_otherdomain;
    mcl->args[0] = va;
    mcl->args[1] = new_val;
    mcl->args[2] = 0;
    mcl->args[3] = flags;
    mcl->args[4] = domid;
}

*/

/*
 * wapper function of hypercalls 
 */

static inline int
HYPERVISOR_yield(
        void)
{
        int rc = HYPERVISOR_sched_op(SCHEDOP_yield, 0);

        return rc;
}

static inline int
HYPERVISOR_block(
        void)
{
        int rc = HYPERVISOR_sched_op(SCHEDOP_block, 0);

        return rc;
}



static inline int
HYPERVISOR_shutdown(
        unsigned int reason)
{

        int rc = HYPERVISOR_sched_op(SCHEDOP_shutdown, reason );

/* x86 case..
        struct sched_shutdown sched_shutdown = {
                .reason = reason
        };

        int rc = HYPERVISOR_sched_op(SCHEDOP_shutdown, &sched_shutdown);

        if (rc == -ENOSYS)
                rc = HYPERVISOR_sched_op_compat(SCHEDOP_shutdown, reason);
 */
        return rc;
}

/* x86 case..
static inline int
HYPERVISOR_poll(
        evtchn_port_t *ports, unsigned int nr_ports, u64 timeout)
{

        struct sched_poll sched_poll = {
                .ports = ports,
                .nr_ports = nr_ports,
                .timeout = jiffies_to_st(timeout)
        };

        int rc = HYPERVISOR_sched_op(SCHEDOP_poll, &sched_poll);

        if (rc == -ENOSYS)
                rc = HYPERVISOR_sched_op_compat(SCHEDOP_yield, 0);
        return rc;
}
*/

#ifdef CONFIG_GCOV_XEN
static inline int HYPERVISOR_gcov_op(unsigned int cmd, void *arg, void *arg2)
{
	register long a0 __asm__("r0") = (long)cmd;
	register long a1 __asm__("r1") = (long)arg;
	register long a2 __asm__("r2") = (long)arg2;
	
	register long f0 __asm__("r7") = __HYPERVISOR_do_gcov_op;
	register long r0 __asm__("r0");
	
	__asm__ __volatile__(
			HYPERCALL_INSTR
			: "=r"(r0)
			: "0"(a0), "r"(a1), "r"(a2), "r"(f0)
			: "memory");

	return (int)r0;
}
#endif
#endif

