/*
 * Copyright (c) 2006-2009 NVIDIA Corporation.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * Neither the name of the NVIDIA Corporation nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */


#ifndef INCLUDED_NVASSERT_H
#define INCLUDED_NVASSERT_H

#include "nvcommon.h"

#if defined(__cplusplus)
extern "C"
{
#endif

/** NvOsBreakPoint - break into debugger.
 * @param  file is the file name (usually from the built-in __FILE__ macro)
 * in which the debug assertion message is to refer. If NULL, no debug assertion
 * message is printed.
 * @param line is the line number within 'file' at which the assertion occurred.
 * @param condition is the assertion condition that failed. If NULL, no condition
 * string will be displayed.
 */
void
NvOsBreakPoint(const char* file, NvU32 line, const char* condition);

/**
 * Macro to break into debugger without printing an assertion message.
 */
#define NV_OS_BREAK_POINT() NvOsBreakPoint(NULL, 0, NULL)


/**
 * Runtime condition check with break into debugger if the assert fails.
 * Compiles out in release builds.
 *
 * We provide two variants of assert: one that prints out the failing assert
 * condition (the "#x" string in the macro), and another that doesn't.  By
 * default, we print the condition string in x86 builds only.  The assumption
 * is that x86 systems have boatloads of memory and can afford to put all these
 * extra strings in the binary, whereas other systems are our target systems
 * and tend to have less memory available.  Also, we want to be careful about
 * anything that might make it take too long to transfer system images over to
 * the target system.
 *
 * We also allow individual developers to override this default behavior,
 * either globally or on a per-source-file basis.  To do this, set
 * NV_ASSERT_PROVIDE_CONDITION_STRING to either 0 or 1, either in your own
 * source code, in your own makefile, or by uncommenting the lines below.
 */

#if !defined(NV_ASSERT)
#if NV_DEBUG

// Uncomment me to override default assert behavior
//#define NV_ASSERT_PROVIDE_CONDITION_STRING 0
//#define NV_ASSERT_PROVIDE_CONDITION_STRING 1

// Default behavior: provide condition string in x86 builds only
#if !defined(NV_ASSERT_PROVIDE_CONDITION_STRING)
#if NVCPU_IS_X86
#define NV_ASSERT_PROVIDE_CONDITION_STRING 1
#else
#define NV_ASSERT_PROVIDE_CONDITION_STRING 0
#endif
#endif

#if NV_ASSERT_PROVIDE_CONDITION_STRING
#define NV_ASSERT(x) \
    do { \
        if (!(x)) \
        { \
            /* print message and break into the debugger */ \
            NvOsBreakPoint(__FILE__, __LINE__, #x); \
        } \
    } while( 0 )
#else // NV_ASSERT_PROVIDE_CONDITION_STRING
#define NV_ASSERT(x) \
    do { \
        if (!(x)) \
        { \
            /* print message and break into the debugger */ \
            NvOsBreakPoint(__FILE__, __LINE__, NULL); \
        } \
    } while( 0 )
#endif // NV_ASSERT_PROVIDE_CONDITION_STRING

#else // NV_DEBUG
#define NV_ASSERT(x) do {} while(0)
#endif // NV_DEBUG
#endif //!defined(NV_ASSERT)

/**
 * NV_CT_ASSERT: compile-time assert for constant values.
 *
 * This works by declaring a function with an array parameter.  If the
 * assert condition is true, then the array size will be 1, otherwise
 * the array size will be -1, which will generate a compilation error.
 *
 * No code should be generated by this macro.
 *
 * Three levels of macros are needed to properly expand the line number.
 *
 * This macro was taken in spirit from:
 *     //sw/main/drivers/common/inc/nvctassert.h
 */
#define NV_CT_ASSERT( x )            NV_CT_ASSERT_I( x, __LINE__ )
#define NV_CT_ASSERT_I( x,line )     NV_CT_ASSERT_II( x, line )
#define NV_CT_ASSERT_II( x, line ) \
    void compile_time_assertion_failed_in_line_##line( \
        int _compile_time_assertion_failed_in_line_##line[(x) ? 1 : -1])

/**
 * A macro to assert (rather than check) that something succeeded.  The use of
 * this macro is strongly discouraged in any production-worthy code.  It is,
 * however, a step up from ignoring the NvError return code of a function, and
 * it is trivial to use and harmless to your release builds.  If everyone uses
 * it, it also makes it easy to search the tree for missing error handling code.
 *
 * In this macro, we don't worry about the stack space wasted from multiple
 * NvError locals in a single function -- production-quality code shouldn't be
 * using this macro in the first place.
 */
#if NV_DEBUG
#define NV_ASSERT_SUCCESS(expr) \
    do \
    { \
        NvError AssertSuccessError = (expr); \
        NV_ASSERT(AssertSuccessError == NvSuccess); \
    } while (0)
#else
#define NV_ASSERT_SUCCESS(expr) \
    do \
    { \
        (void)(expr); \
    } while (0)
#endif

#if defined(__cplusplus)
}
#endif

#endif // INCLUDED_NVASSERT_H
