/*
 * Copyright (c) 2006-2009 NVIDIA Corporation.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * Neither the name of the NVIDIA Corporation nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/** 
 * @file
 * <b>NVIDIA Tegra ODM Kit:
 *         Scroll Wheel Interface</b>
 *
 * @b Description: Defines the ODM adaptation interface for scroll wheel devices.
 * 
 */

#ifndef INCLUDED_NVODM_SCROLLWHEEL_H
#define INCLUDED_NVODM_SCROLLWHEEL_H

#if defined(__cplusplus)
extern "C"
{
#endif

#include "nvodm_services.h"

/**
 * @defgroup nvodm_scrollwheel Scroll Wheel Adaptation Interface
 * 
 * This is the scroll wheel ODM adaptation interface. 
 * 
 * @ingroup nvodm_adaptation
 * @{
 */

/**
 * Defines an opaque handle that exists for each scroll wheel device in the
 * system, each of which is defined by the customer implementation.
 */
typedef struct NvOdmScrollWheelRec *NvOdmScrollWheelHandle;

/**
 * Defines the events generated by the scroll wheel.
 */
typedef enum
{
    /// Indicates no event.
    NvOdmScrollWheelEvent_None = 0,
    
    /// Indicates the scroll up event.
    NvOdmScrollWheelEvent_RotateClockWise = 0x1,
    
    /// Indicates the scroll down event.
    NvOdmScrollWheelEvent_RotateAntiClockWise = 0x2,
    
    /// Indicates the key press event.
    NvOdmScrollWheelEvent_Press = 0x4,
    
    /// Indicates the release event.
    NvOdmScrollWheelEvent_Release = 0x8,

    /// Indicates the scroll up event.
    NvOdmScrollWheelEvent_Up = 0x10,

    /// Indicates the scroll down event.
    NvOdmScrollWheelEvent_Down = 0x20,

    /// Indicates the scroll right event.
    NvOdmScrollWheelEvent_Right = 0x40,

    /// Indicates the scroll left event.
    NvOdmScrollWheelEvent_Left = 0x80,
    
    NvOdmScrollWheelEvent_Force32 = 0x7FFFFFFF
} NvOdmScrollWheelEvent;



/**
 * Gets a handle to the scroll wheel device.
 *
 * @param hSema A handle to the semaphore to be signalled when 
 * there is an event.
 * @param KeyEvents Specifies the ORed version of all the events for which
 * the client wants to register.
 *
 * @return A handle to the scroll wheel device.
 */
NvOdmScrollWheelHandle 
NvOdmScrollWheelOpen( 
    NvOdmOsSemaphoreHandle hSema,
    NvOdmScrollWheelEvent KeyEvents);

/**
 * Closes the scroll wheel handle. 
 *
 * @param hOdmScrollWheel The scroll wheel handle to be closed.
 */
void NvOdmScrollWheelClose(NvOdmScrollWheelHandle hOdmScrollWheel);


/**
 * Gets the pending event information. 
 *
 * This API should be called when the semaphore
 * that the client is waiting on is signalled.
 *
 * @param hOdmScrollWheel A handle to the scroll wheel.
 * @return The type of event. If there are no pending events,
 * returns the 'none' event. If there are more than one events, 
 * returns the ORed version of the events.
 *
 */
NvOdmScrollWheelEvent NvOdmScrollWheelGetEvent(NvOdmScrollWheelHandle hOdmScrollWheel);

#if defined(__cplusplus)
}
#endif

/** @} */

#endif // INCLUDED_NVODM_SCROLLWHEEL_H
