/****************************************************************************
 * Driver for Solarflare network controllers
 *           (including support for SFE4001 10GBT NIC)
 *
 * Copyright 2005-2006: Fen Systems Ltd.
 * Copyright 2006-2008: Solarflare Communications Inc,
 *                      9501 Jeronimo Road, Suite 250,
 *                      Irvine, CA 92618, USA
 *
 * Initially developed by Michael Brown <mbrown@fensystems.co.uk>
 * Maintained by Solarflare Communications <linux-net-drivers@solarflare.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 ****************************************************************************
 */

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/dcache.h>
#include <linux/seq_file.h>
#include "net_driver.h"
#include "efx.h"
#include "debugfs.h"
#include "falcon.h"

#ifndef PRIu64
#	if (BITS_PER_LONG == 64)
#		define PRIu64 "lu"
#	else
#		define PRIu64 "llu"
#	endif
#endif

static void efx_debugfs_remove(struct proc_dir_entry *entry)
{
	if (entry)
		remove_proc_entry(entry->name, entry->parent);
}
#define debugfs_remove efx_debugfs_remove

#define debugfs_create_dir proc_mkdir
#define debugfs_create_symlink proc_symlink


/* Parameter definition bound to a structure - each file has one of these */
struct efx_debugfs_bound_param {
	const struct efx_debugfs_parameter *param;
	void *structure;
};


/* Maximum length for a name component or symlink target */
#define EFX_DEBUGFS_NAME_LEN 32


/* Top-level debug directory ([/sys/kernel]/debug/sfc) */
static struct dentry *efx_debug_root;

/* "cards" directory ([/sys/kernel]/debug/sfc/cards) */
static struct dentry *efx_debug_cards;


/* Sequential file interface to bound parameters */

static int efx_debugfs_seq_show(struct seq_file *file, void *v)
{
	struct proc_dir_entry *entry = (struct proc_dir_entry *)file->private;
	struct efx_debugfs_parameter *param =
		(struct efx_debugfs_parameter *)entry->data;
	void *structure = (void *)entry->read_proc;

	if (!structure)
		return -EIO;

	return param->reader(file, structure + param->offset);
}

static int efx_debugfs_open(struct inode *inode, struct file *file)
{
	return single_open(file, efx_debugfs_seq_show, PROC_I(inode)->pde);
}


static struct file_operations efx_debugfs_file_ops = {
	.owner   = THIS_MODULE,
	.open    = efx_debugfs_open,
	.read    = seq_read,
	.llseek  = seq_lseek,
	.release = seq_release
};


void efx_fini_debugfs_child(struct proc_dir_entry *dir, const char *name)
{
	remove_proc_entry(name, dir);
}

/*
 * Remove a debugfs directory.
 *
 * This removes the named parameter-files and sym-links from the
 * directory, and the directory itself.  It does not do any recursion
 * to subdirectories.
 */
static void efx_fini_debugfs_dir(struct dentry *dir,
				 struct efx_debugfs_parameter *params,
				 const char *const *symlink_names)
{
	if (!dir)
		return;

	while (params->name) {
		efx_fini_debugfs_child(dir, params->name);
		params++;
	}
	while (symlink_names && *symlink_names) {
		efx_fini_debugfs_child(dir, *symlink_names);
		symlink_names++;
	}
	debugfs_remove(dir);
}

/* Functions for printing various types of parameter. */

int efx_debugfs_read_uint(struct seq_file *file, void *data)
{
	return seq_printf(file, "%#x\n", *(unsigned int *)data);
}

int efx_debugfs_read_int(struct seq_file *file, void *data)
{
	return seq_printf(file, "%d\n", *(int *)data);
}

int efx_debugfs_read_atomic(struct seq_file *file, void *data)
{
	unsigned int value = atomic_read((atomic_t *) data);

	return seq_printf(file, "%#x\n", value);
}

int efx_debugfs_read_dword(struct seq_file *file, void *data)
{
	unsigned int value = EFX_DWORD_FIELD(*(efx_dword_t *) data,
					     EFX_DWORD_0);

	return seq_printf(file, "%#x\n", value);
}

static int efx_debugfs_read_int_mode(struct seq_file *file, void *data)
{
	unsigned int value = *(enum efx_int_mode *) data;

	return seq_printf(file, "%d => %s\n", value,
			  STRING_TABLE_LOOKUP(value, efx_interrupt_mode));
}

#define EFX_INT_MODE_PARAMETER(container_type, parameter)		\
	EFX_PARAMETER(container_type, parameter,			\
		      enum efx_int_mode, efx_debugfs_read_int_mode)

static int efx_debugfs_read_loop_mode(struct seq_file *file, void *data)
{
	unsigned int value = *(enum efx_loopback_mode *)data;

	return seq_printf(file, "%d => %s\n", value,
			  STRING_TABLE_LOOKUP(value, efx_loopback_mode));
}

#define EFX_LOOPBACK_MODE_PARAMETER(container_type, parameter)		\
	EFX_PARAMETER(container_type, parameter,			\
		      enum efx_loopback_mode, efx_debugfs_read_loop_mode)

static int efx_debugfs_read_phy_type(struct seq_file *file, void *data)
{
	unsigned int value = *(enum phy_type *) data;

	return seq_printf(file, "%d => %s\n", value,
			  STRING_TABLE_LOOKUP(value, efx_phy_type));
}

#define EFX_PHY_TYPE_PARAMETER(container_type, parameter)		\
	EFX_PARAMETER(container_type, parameter,			\
		      enum phy_type, efx_debugfs_read_phy_type)

int efx_debugfs_read_string(struct seq_file *file, void *data)
{
	return seq_puts(file, (const char *)data);
}


/**
 * efx_init_debugfs_files - create parameter-files in a debugfs directory
 * @parent:		Containing directory
 * @params:		Pointer to zero-terminated parameter definition array
 * @structure:		Structure containing parameters
 *
 * Add parameter-files to the given debugfs directory.  Return a
 * negative error code or 0 on success.
 */
static int efx_init_debugfs_files(struct dentry *parent,
				  struct efx_debugfs_parameter *params,
				  void *structure)
{
	struct efx_debugfs_parameter *param = params;

	while (param->name) {
		struct dentry *entry;
		entry = create_proc_entry(param->name, S_IRUGO, parent);
		if (!entry)
			goto err;
		/*
		 * We have no good way to free a binding created here.
		 * However, once we install our file_operations the
		 * read_proc pointer becomes redundant and we can
		 * abuse it as a structure pointer.
		 */
		entry->data = param;
		entry->read_proc = NULL;
		smp_wmb();
		entry->proc_fops = &efx_debugfs_file_ops;
		smp_wmb();
		entry->read_proc = (read_proc_t *) structure;

		param++;
	}

	return 0;

 err:
	while (param != params) {
		param--;
		efx_fini_debugfs_child(parent, param->name);
	}
	return -ENOMEM;
}

/**
 * efx_init_debugfs_netdev - create debugfs sym-links for net device
 * @net_dev:		Net device
 *
 * Create sym-links named after @net_dev to the debugfs directories for
 * the corresponding NIC and  port.  Return a negative error code or 0 on
 * success.  The sym-links must be cleaned up using
 * efx_fini_debugfs_netdev().
 */
int efx_init_debugfs_netdev(struct net_device *net_dev)
{
	struct efx_nic *efx = net_dev->priv;
	char name[EFX_DEBUGFS_NAME_LEN];
	char target[EFX_DEBUGFS_NAME_LEN];
	size_t len;

	if (snprintf(name, sizeof(name), "nic_%s", net_dev->name) >=
	    sizeof(name))
		return -ENAMETOOLONG;
	if (snprintf(target, sizeof(target), "cards/%s", pci_name(efx->pci_dev))
	    >= sizeof(target))
		return -ENAMETOOLONG;
	efx->debug_symlink = debugfs_create_symlink(name,
						    efx_debug_root, target);
	if (!efx->debug_symlink)
		return -ENOMEM;

	if (snprintf(name, sizeof(name), "if_%s", net_dev->name) >=
	    sizeof(name))
		return -ENAMETOOLONG;
	len = snprintf(target, sizeof(target),
		       "cards/%s/port0", pci_name(efx->pci_dev));
	if (len >= sizeof(target))
		return -ENAMETOOLONG;
	efx->debug_port_symlink = debugfs_create_symlink(name,
							 efx_debug_root,
							 target);
	if (!efx->debug_port_symlink)
		return -ENOMEM;

	return 0;
}

/**
 * efx_fini_debugfs_netdev - remove debugfs sym-links for net device
 * @net_dev:		Net device
 *
 * Remove sym-links created for @net_dev by efx_init_debugfs_netdev().
 */
void efx_fini_debugfs_netdev(struct net_device *net_dev)
{
	struct efx_nic *efx = net_dev->priv;

	debugfs_remove(efx->debug_port_symlink);
	efx->debug_port_symlink = NULL;
	debugfs_remove(efx->debug_symlink);
	efx->debug_symlink = NULL;
}

/* Per-port parameters */
static struct efx_debugfs_parameter efx_debugfs_port_parameters[] = {
	EFX_NAMED_PARAMETER(enabled, struct efx_nic, port_enabled,
			    int, efx_debugfs_read_int),
	EFX_INT_PARAMETER(struct efx_nic, rx_checksum_enabled),
	EFX_ATOMIC_PARAMETER(struct efx_nic, netif_stop_count),
	EFX_INT_PARAMETER(struct efx_nic, link_up),
	EFX_UINT_PARAMETER(struct efx_nic, link_options),
	EFX_INT_PARAMETER(struct efx_nic, promiscuous),
	EFX_UINT_PARAMETER(struct efx_nic, loopback_modes),
	EFX_LOOPBACK_MODE_PARAMETER(struct efx_nic, loopback_mode),
	EFX_PHY_TYPE_PARAMETER(struct efx_nic, phy_type),
	EFX_NAMED_PARAMETER(phy_id, struct efx_nic, mii.phy_id,
			    int, efx_debugfs_read_int),
	EFX_UINT_PARAMETER(struct efx_nic, n_link_state_changes),
	{NULL},
};

/**
 * efx_init_debugfs_port - create debugfs directory for port
 * @efx:		Efx NIC
 *
 * Create a debugfs directory containing parameter-files for @efx.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efx_fini_debugfs_port().
 */
int efx_init_debugfs_port(struct efx_nic *efx)
{
	int rc;

	/* Create directory */
	efx->debug_port_dir = debugfs_create_dir("port0", efx->debug_dir);
	if (!efx->debug_port_dir)
		return -ENOMEM;

	/* Create files */
	rc = efx_init_debugfs_files(efx->debug_port_dir,
				    efx_debugfs_port_parameters,
				    (void *)efx);
	if (rc)
		efx_fini_debugfs_port(efx);

	return rc;
}

/**
 * efx_fini_debugfs_port - remove debugfs directory for port
 * @efx:		Efx NIC
 *
 * Remove directory created for @efx by efx_init_debugfs_port().
 */
void efx_fini_debugfs_port(struct efx_nic *efx)
{
	efx_fini_debugfs_dir(efx->debug_port_dir,
			     efx_debugfs_port_parameters, NULL);
	efx->debug_port_dir = NULL;
}

/**
 * efx_extend_debugfs_port - add parameter-files to directory for port
 * @efx:		Efx NIC
 * @structure:		Structure containing parameters
 * @params:		Pointer to zero-terminated parameter definition array
 *
 * Add parameter-files to the debugfs directory for @efx.  Return
 * a negative error code or 0 on success.  This is intended for
 * PHY-specific parameters.  The files must be cleaned up using
 * efx_trim_debugfs_port().
 */
int efx_extend_debugfs_port(struct efx_nic *efx,
			    void *structure,
			    struct efx_debugfs_parameter *params)
{
	return efx_init_debugfs_files(efx->debug_port_dir, params, structure);
}

/**
 * efx_trim_debugfs_port - remove parameter-files from directory for port
 * @efx:		Efx NIC
 * @params:		Pointer to zero-terminated parameter definition array
 *
 * Remove parameter-files previously added to the debugfs directory
 * for @efx using efx_extend_debugfs_port().
 */
void efx_trim_debugfs_port(struct efx_nic *efx,
			   struct efx_debugfs_parameter *params)
{
	struct dentry *dir = efx->debug_port_dir;

	if (dir) {
		struct efx_debugfs_parameter *field;
		for (field = params; field->name; field++)
			efx_fini_debugfs_child(dir, field->name);
	}
}

/* Per-TX-queue parameters */
static struct efx_debugfs_parameter efx_debugfs_tx_queue_parameters[] = {
	EFX_UINT_PARAMETER(struct efx_tx_queue, insert_count),
	EFX_UINT_PARAMETER(struct efx_tx_queue, write_count),
	EFX_UINT_PARAMETER(struct efx_tx_queue, read_count),
	EFX_INT_PARAMETER(struct efx_tx_queue, stopped),
	{NULL},
};

static void efx_fini_debugfs_tx_queue(struct efx_tx_queue *tx_queue);

/**
 * efx_init_debugfs_tx_queue - create debugfs directory for TX queue
 * @tx_queue:		Efx TX queue
 *
 * Create a debugfs directory containing parameter-files for @tx_queue.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efx_fini_debugfs_tx_queue().
 */
static int efx_init_debugfs_tx_queue(struct efx_tx_queue *tx_queue)
{
	char name[EFX_DEBUGFS_NAME_LEN];
	char target[EFX_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), EFX_TX_QUEUE_NAME(tx_queue))
	    >= sizeof(name))
		goto err_len;
	tx_queue->debug_dir = debugfs_create_dir(name,
						 tx_queue->efx->debug_dir);
	if (!tx_queue->debug_dir)
		goto err_mem;

	/* Create files */
	rc = efx_init_debugfs_files(tx_queue->debug_dir,
				    efx_debugfs_tx_queue_parameters,
				    (void *)tx_queue);
	if (rc)
		goto err;

	/* Create symlink to channel */
	if (snprintf(target, sizeof(target),
		     "../" EFX_CHANNEL_NAME(tx_queue->channel)) >=
	    sizeof(target))
		goto err_len;
	if (!debugfs_create_symlink("channel", tx_queue->debug_dir, target))
		goto err_mem;

	/* Create symlink to port */
	if (!debugfs_create_symlink("port", tx_queue->debug_dir, "../port0"))
		goto err_mem;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efx_fini_debugfs_tx_queue(tx_queue);
	return rc;
}

/**
 * efx_fini_debugfs_tx_queue - remove debugfs directory for TX queue
 * @tx_queue:		Efx TX queue
 *
 * Remove directory created for @tx_queue by efx_init_debugfs_tx_queue().
 */
static void efx_fini_debugfs_tx_queue(struct efx_tx_queue *tx_queue)
{
	static const char *const symlink_names[] = {
		"channel", "port", NULL
	};

	efx_fini_debugfs_dir(tx_queue->debug_dir,
			     efx_debugfs_tx_queue_parameters, symlink_names);
	tx_queue->debug_dir = NULL;
}

/* Per-RX-queue parameters */
static struct efx_debugfs_parameter efx_debugfs_rx_queue_parameters[] = {
	EFX_INT_PARAMETER(struct efx_rx_queue, added_count),
	EFX_INT_PARAMETER(struct efx_rx_queue, removed_count),
	EFX_UINT_PARAMETER(struct efx_rx_queue, max_fill),
	EFX_UINT_PARAMETER(struct efx_rx_queue, fast_fill_trigger),
	EFX_UINT_PARAMETER(struct efx_rx_queue, fast_fill_limit),
	EFX_UINT_PARAMETER(struct efx_rx_queue, min_fill),
	EFX_UINT_PARAMETER(struct efx_rx_queue, min_overfill),
	EFX_UINT_PARAMETER(struct efx_rx_queue, alloc_page_count),
	EFX_UINT_PARAMETER(struct efx_rx_queue, alloc_skb_count),
	EFX_UINT_PARAMETER(struct efx_rx_queue, slow_fill_count),
	{NULL},
};

static void efx_fini_debugfs_rx_queue(struct efx_rx_queue *rx_queue);

/**
 * efx_init_debugfs_rx_queue - create debugfs directory for RX queue
 * @rx_queue:		Efx RX queue
 *
 * Create a debugfs directory containing parameter-files for @rx_queue.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efx_fini_debugfs_rx_queue().
 */
static int efx_init_debugfs_rx_queue(struct efx_rx_queue *rx_queue)
{
	char name[EFX_DEBUGFS_NAME_LEN];
	char target[EFX_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), EFX_RX_QUEUE_NAME(rx_queue))
	    >= sizeof(name))
		goto err_len;
	rx_queue->debug_dir = debugfs_create_dir(name,
						 rx_queue->efx->debug_dir);
	if (!rx_queue->debug_dir)
		goto err_mem;

	/* Create files */
	rc = efx_init_debugfs_files(rx_queue->debug_dir,
				    efx_debugfs_rx_queue_parameters,
				    (void *)rx_queue);
	if (rc)
		goto err;

	/* Create symlink to channel */
	if (snprintf(target, sizeof(target),
		     "../" EFX_CHANNEL_NAME(rx_queue->channel)) >=
	    sizeof(target))
		goto err_len;
	if (!debugfs_create_symlink("channel", rx_queue->debug_dir, target))
		goto err_mem;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efx_fini_debugfs_rx_queue(rx_queue);
	return rc;
}

/**
 * efx_fini_debugfs_rx_queue - remove debugfs directory for RX queue
 * @rx_queue:		Efx RX queue
 *
 * Remove directory created for @rx_queue by efx_init_debugfs_rx_queue().
 */
static void efx_fini_debugfs_rx_queue(struct efx_rx_queue *rx_queue)
{
	const char *const symlink_names[] = {
		"channel", NULL
	};

	efx_fini_debugfs_dir(rx_queue->debug_dir,
			     efx_debugfs_rx_queue_parameters, symlink_names);
	rx_queue->debug_dir = NULL;
}

/* Per-channel parameters */
static struct efx_debugfs_parameter efx_debugfs_channel_parameters[] = {
	EFX_INT_PARAMETER(struct efx_channel, enabled),
	EFX_INT_PARAMETER(struct efx_channel, irq),
	EFX_UINT_PARAMETER(struct efx_channel, has_interrupt),
	EFX_UINT_PARAMETER(struct efx_channel, irq_moderation),
	EFX_UINT_PARAMETER(struct efx_channel, eventq_read_ptr),
	EFX_UINT_PARAMETER(struct efx_channel, n_rx_tobe_disc),
	EFX_UINT_PARAMETER(struct efx_channel, n_rx_ip_frag_err),
	EFX_UINT_PARAMETER(struct efx_channel, n_rx_ip_hdr_chksum_err),
	EFX_UINT_PARAMETER(struct efx_channel, n_rx_tcp_udp_chksum_err),
	EFX_UINT_PARAMETER(struct efx_channel, n_rx_frm_trunc),
	EFX_UINT_PARAMETER(struct efx_channel, n_rx_overlength),
	EFX_UINT_PARAMETER(struct efx_channel, n_skbuff_leaks),
	EFX_INT_PARAMETER(struct efx_channel, rx_alloc_level),
	EFX_INT_PARAMETER(struct efx_channel, rx_alloc_push_pages),
	EFX_INT_PARAMETER(struct efx_channel, rx_alloc_pop_pages),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_merges),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_bursts),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_slow_start),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_misorder),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_too_many),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_new_stream),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_drop_idle),
	EFX_UINT_PARAMETER(struct efx_channel, ssr.n_drop_closed),
	{NULL},
};

static void efx_fini_debugfs_channel(struct efx_channel *channel);

/**
 * efx_init_debugfs_channel - create debugfs directory for channel
 * @channel:		Efx channel
 *
 * Create a debugfs directory containing parameter-files for @channel.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efx_fini_debugfs_channel().
 */
static int efx_init_debugfs_channel(struct efx_channel *channel)
{
	char name[EFX_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), EFX_CHANNEL_NAME(channel))
	    >= sizeof(name))
		goto err_len;
	channel->debug_dir = debugfs_create_dir(name, channel->efx->debug_dir);
	if (!channel->debug_dir)
		goto err_mem;

	/* Create files */
	rc = efx_init_debugfs_files(channel->debug_dir,
				    efx_debugfs_channel_parameters,
				    (void *)channel);
	if (rc)
		goto err;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efx_fini_debugfs_channel(channel);
	return rc;
}

/**
 * efx_fini_debugfs_channel - remove debugfs directory for channel
 * @channel:		Efx channel
 *
 * Remove directory created for @channel by efx_init_debugfs_channel().
 */
static void efx_fini_debugfs_channel(struct efx_channel *channel)
{
	efx_fini_debugfs_dir(channel->debug_dir,
			     efx_debugfs_channel_parameters, NULL);
	channel->debug_dir = NULL;
}

/* Per-NIC parameters */
static struct efx_debugfs_parameter efx_debugfs_nic_parameters[] = {
	EFX_INT_PARAMETER(struct efx_nic, legacy_irq),
	EFX_INT_PARAMETER(struct efx_nic, rss_queues),
	EFX_UINT_PARAMETER(struct efx_nic, rx_buffer_len),
	EFX_INT_MODE_PARAMETER(struct efx_nic, interrupt_mode),
	{.name = "hardware_desc",
	 .offset = 0,
	 .reader = falcon_debugfs_read_hardware_desc},
	{NULL},
};

/* Per-NIC error counts */
static struct efx_debugfs_parameter efx_debugfs_nic_error_parameters[] = {
	EFX_ATOMIC_PARAMETER(struct efx_nic_errors, missing_event),
	EFX_ATOMIC_PARAMETER(struct efx_nic_errors, rx_reset),
	EFX_ATOMIC_PARAMETER(struct efx_nic_errors, rx_desc_fetch),
	EFX_ATOMIC_PARAMETER(struct efx_nic_errors, tx_desc_fetch),
	EFX_ATOMIC_PARAMETER(struct efx_nic_errors, spurious_tx),
	{NULL},
};

/**
 * efx_init_debugfs_channels - create debugfs directories for NIC channels
 * @efx:		Efx NIC
 *
 * Create subdirectories of @efx's debugfs directory for all the
 * channels, RX queues and TX queues used by this driver.  Return a
 * negative error code or 0 on success.  The subdirectories must be
 * cleaned up using efx_fini_debugfs_channels().
 */
int efx_init_debugfs_channels(struct efx_nic *efx)
{
	struct efx_channel *channel;
	struct efx_rx_queue *rx_queue;
	struct efx_tx_queue *tx_queue;
	int rc;

	efx_for_each_channel(channel, efx) {
		rc = efx_init_debugfs_channel(channel);
		if (rc)
			goto err;
	}

	efx_for_each_rx_queue(rx_queue, efx) {
		rc = efx_init_debugfs_rx_queue(rx_queue);
		if (rc)
			goto err;
	}

	efx_for_each_tx_queue(tx_queue, efx) {
		rc = efx_init_debugfs_tx_queue(tx_queue);
		if (rc)
			goto err;
	}

	return 0;

 err:
	efx_fini_debugfs_channels(efx);
	return rc;
}

/**
 * efx_fini_debugfs_channels - remove debugfs directories for NIC queues
 * @efx:		Efx NIC
 *
 * Remove subdirectories of @efx's debugfs directory created by
 * efx_init_debugfs_channels().
 */
void efx_fini_debugfs_channels(struct efx_nic *efx)
{
	struct efx_channel *channel;
	struct efx_rx_queue *rx_queue;
	struct efx_tx_queue *tx_queue;

	efx_for_each_tx_queue(tx_queue, efx)
		efx_fini_debugfs_tx_queue(tx_queue);

	efx_for_each_rx_queue(rx_queue, efx)
		efx_fini_debugfs_rx_queue(rx_queue);

	efx_for_each_channel(channel, efx)
		efx_fini_debugfs_channel(channel);
}

/**
 * efx_init_debugfs_nic - create debugfs directory for NIC
 * @efx:		Efx NIC
 *
 * Create debugfs directory containing parameter-files for @efx,
 * and a subdirectory "errors" containing per-NIC error counts.
 * Return a negative error code or 0 on success.  The directories
 * must be cleaned up using efx_fini_debugfs_nic().
 */
int efx_init_debugfs_nic(struct efx_nic *efx)
{
	int rc;

	/* Create directory */
	efx->debug_dir = debugfs_create_dir(pci_name(efx->pci_dev),
					    efx_debug_cards);
	if (!efx->debug_dir)
		goto err_mem;

	/* Create errors directory */
	efx->errors.debug_dir = debugfs_create_dir("errors", efx->debug_dir);
	if (!efx->errors.debug_dir)
		goto err_mem;

	/* Create files */
	rc = efx_init_debugfs_files(efx->debug_dir,
				    efx_debugfs_nic_parameters, (void *)efx);
	if (rc)
		goto err;
	rc = efx_init_debugfs_files(efx->errors.debug_dir,
				    efx_debugfs_nic_error_parameters,
				    (void *)&efx->errors);
	if (rc)
		goto err;

	return 0;

 err_mem:
	rc = -ENOMEM;
 err:
	efx_fini_debugfs_nic(efx);
	return rc;
}

/**
 * efx_fini_debugfs_nic - remove debugfs directories for NIC
 * @efx:		Efx NIC
 *
 * Remove debugfs directories created for @efx by efx_init_debugfs_nic().
 */
void efx_fini_debugfs_nic(struct efx_nic *efx)
{
	efx_fini_debugfs_dir(efx->errors.debug_dir,
			     efx_debugfs_nic_error_parameters, NULL);
	efx->errors.debug_dir = NULL;
	efx_fini_debugfs_dir(efx->debug_dir, efx_debugfs_nic_parameters, NULL);
	efx->debug_dir = NULL;
}

/**
 * efx_init_debugfs - create debugfs directories for sfc driver
 *
 * Create debugfs directories "sfc" and "sfc/cards".  This must be
 * called before any of the other functions that create debugfs
 * directories.  Return a negative error code or 0 on success.  The
 * directories must be cleaned up using efx_fini_debugfs().
 */
int efx_init_debugfs(void)
{
	/* Create top-level directory */
	efx_debug_root = proc_mkdir("sfc", proc_root_driver);
	if (!efx_debug_root)
		goto err;

	/* Create "cards" directory */
	efx_debug_cards = debugfs_create_dir("cards", efx_debug_root);
	if (!efx_debug_cards)
		goto err;

	return 0;

 err:
	efx_fini_debugfs();
	return -ENOMEM;
}

/**
 * efx_fini_debugfs - remove debugfs directories for sfc driver
 *
 * Remove directories created by efx_init_debugfs().
 */
void efx_fini_debugfs(void)
{
	remove_proc_entry("sfc", proc_root_driver);
	debugfs_remove(efx_debug_cards);
	efx_debug_cards = NULL;
	debugfs_remove(efx_debug_root);
	efx_debug_root = NULL;
}
