/****************************************************************************
 * Driver for Solarflare network controllers -
 *          resource management for Xen backend, OpenOnload, etc
 *           (including support for SFE4001 10GBT NIC)
 *
 * This file provides private API of efrm library -- resource handling.
 * This API is not designed for use outside of SFC resource driver.
 *
 * Copyright 2005-2007: Solarflare Communications Inc,
 *                      9501 Jeronimo Road, Suite 250,
 *                      Irvine, CA 92618, USA
 *
 * Developed and maintained by Solarflare Communications:
 *                      <linux-xen-drivers@solarflare.com>
 *                      <onload-dev@solarflare.com>
 *
 * Certain parts of the driver were implemented by
 *          Alexandra Kossovsky <Alexandra.Kossovsky@oktetlabs.ru>
 *          OKTET Labs Ltd, Russia,
 *          http://oktetlabs.ru, <info@oktetlabs.ru>
 *          by request of Solarflare Communications
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 ****************************************************************************
 */

#ifndef __CI_EFRM_PRIVATE_H__
#define __CI_EFRM_PRIVATE_H__

#include <ci/efrm/resource.h>
#include <ci/efrm/driver_private.h>
#include <ci/efrm/sysdep.h>
#include <ci/efrm/debug.h>

/*--------------------------------------------------------------------
 *
 * create resource managers
 *
 *--------------------------------------------------------------------*/

/*! Create a resource manager for various types of resources
 */
extern int
efrm_create_iobufset_resource_manager(struct efrm_resource_manager **out);

extern int
efrm_create_filter_resource_manager(struct efrm_resource_manager **out);

extern int
efrm_create_vi_resource_manager(struct efrm_resource_manager **out,
				const struct vi_resource_dimensions *);

/*--------------------------------------------------------------------
 *
 * efrm_resource_handle_t handling
 *
 *--------------------------------------------------------------------*/

/*! Initialize an area of memory to be used as a resource */
static inline void efrm_resource_init(struct efrm_resource *rs,
				      int type, int instance)
{
	EFRM_ASSERT(instance >= 0);
	EFRM_ASSERT(type >= 0 && type < EFRM_RESOURCE_NUM);
	atomic_set(&rs->rs_ref_count, 1);
	rs->rs_handle.handle = (type << 28u) |
		(((unsigned)jiffies & 0xfff) << 16) | instance;
}

/*--------------------------------------------------------------------
 *
 * Instance pool management
 *
 *--------------------------------------------------------------------*/

/*! Allocate instance pool. Use kfifo_vfree to destroy it. */
static inline int
efrm_kfifo_id_ctor(struct kfifo **ids_out,
		   unsigned int base, unsigned int limit, spinlock_t *lock)
{
	unsigned int i;
	struct kfifo *ids;
	unsigned char *buffer;
#ifndef TCP_CHIMNEY_SUPPORT
	unsigned int size = roundup_pow_of_two((limit - base) * sizeof(int));
#else
        /* ### TODO - Linux kfifos really are a power of two, sysdep_ci2linux
                      does ci_fifo2's, which only actually hold 2^n - 1.
                      We need to double buffer size, not add one, because
                      ci_fifo2 can only be a power of two. */
	unsigned int size = roundup_pow_of_two((limit - base) * 2 * sizeof(int));
#endif

	EFRM_ASSERT(base <= limit);
	buffer = vmalloc(size);
	ids = kfifo_init(buffer, size, GFP_KERNEL, lock);
	if (IS_ERR(ids))
		return PTR_ERR(ids);
	for (i = base; i < limit; i++)
		EFRM_VERIFY_EQ(__kfifo_put(ids, (unsigned char *)&i,
					   sizeof(i)), sizeof(i));

	*ids_out = ids;
	return 0;
}

/*--------------------------------------------------------------------
 *
 * Various private functions
 *
 *--------------------------------------------------------------------*/

/*! Initialize the fields in the provided resource manager memory area
 *   \param rm         The area of memory to be initialized
 *   \param dtor       A method to destroy the resource manager
 *   \param name       A Textual name for the resource manager
 *   \param type       The type of resource managed
 *   \param initial_table_size Initial size of the ID table
 *   \param auto_destroy Destroy resource manager on driver onload iff true
 *
 * A default table size is provided if the value 0 is provided.
 */
extern int
efrm_resource_manager_ctor(struct efrm_resource_manager *rm,
			   void (*dtor)(struct efrm_resource_manager *),
			   const char *name, unsigned type,
			   int initial_table_size);

extern void efrm_resource_manager_dtor(struct efrm_resource_manager *rm);

/*! Insert a resource into table in the resource manager.
 *
 * Caller should free the resource if this function returns non-zero.
 */
extern int efrm_resource_manager_insert(struct efrm_resource *rs);

#endif /* __CI_EFRM_PRIVATE_H__ */
