/****************************************************************************
 * Driver for Solarflare network controllers
 *           (including support for SFE4001 10GBT NIC)
 *
 * Copyright 2005-2006: Fen Systems Ltd.
 * Copyright 2005-2008: Solarflare Communications Inc,
 *                      9501 Jeronimo Road, Suite 250,
 *                      Irvine, CA 92618, USA
 *
 * Initially developed by Michael Brown <mbrown@fensystems.co.uk>
 * Maintained by Solarflare Communications <linux-net-drivers@solarflare.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 ****************************************************************************
 */

#ifndef EFX_KERNEL_COMPAT_H
#define EFX_KERNEL_COMPAT_H

#include <linux/version.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/pci.h>
#include <linux/workqueue.h>
#include <linux/moduleparam.h>
#include <linux/interrupt.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/rtnetlink.h>

/*
 * Kernel backwards compatibility
 *
 * This file provides macros to facilitate backporting the driver.
 */

#ifdef __ia64__
	/* csum_tcpudp_nofold() is extern but not exported */
	#define EFX_NEED_CSUM_TCPUDP_NOFOLD yes
#endif
#ifdef CONFIG_PPC64
	/* __raw_writel and friends are broken on ppc64 */
	#define EFX_NEED_RAW_READ_AND_WRITE_FIX yes
#endif

typedef irqreturn_t (*irq_handler_t)(int, void *, struct pt_regs *);

#define skb_mac_header(skb)	(skb)->mac.raw
#define skb_network_header(skb) (skb)->nh.raw
#define eth_hdr(skb)		((struct ethhdr *)skb_mac_header(skb))
#define tcp_hdr(skb)		(skb)->h.th
#define ip_hdr(skb)		(skb)->nh.iph
#define skb_tail_pointer(skb)   (skb)->tail


#ifdef EFX_NEED_RAW_READ_AND_WRITE_FIX
	#include <asm/io.h>
	static inline void
	efx_raw_writeb(u8 value, volatile void __iomem *addr)
	{
		writeb(value, addr);
	}
	static inline void
	efx_raw_writew(u16 value, volatile void __iomem *addr)
	{
		writew(le16_to_cpu(value), addr);
	}
	static inline void
	efx_raw_writel(u32 value, volatile void __iomem *addr)
	{
		writel(le32_to_cpu(value), addr);
	}
	static inline void
	efx_raw_writeq(u64 value, volatile void __iomem *addr)
	{
		writeq(le64_to_cpu(value), addr);
	}
	static inline u8
	efx_raw_readb(const volatile void __iomem *addr)
	{
		return readb(addr);
	}
	static inline u16
	efx_raw_readw(const volatile void __iomem *addr)
	{
		return cpu_to_le16(readw(addr));
	}
	static inline u32
	efx_raw_readl(const volatile void __iomem *addr)
	{
		return cpu_to_le32(readl(addr));
	}
	static inline u64
	efx_raw_readq(const volatile void __iomem *addr)
	{
		return cpu_to_le64(readq(addr));
	}

	#undef __raw_writeb
	#undef __raw_writew
	#undef __raw_writel
	#undef __raw_writeq
	#undef __raw_readb
	#undef __raw_readw
	#undef __raw_readl
	#undef __raw_readq
	#define __raw_writeb efx_raw_writeb
	#define __raw_writew efx_raw_writew
	#define __raw_writel efx_raw_writel
	#define __raw_writeq efx_raw_writeq
	#define __raw_readb efx_raw_readb
	#define __raw_readw efx_raw_readw
	#define __raw_readl efx_raw_readl
	#define __raw_readq efx_raw_readq
#endif

typedef u32 __wsum;
#define csum_unfold(x) ((__force __wsum) x)

#define DECLARE_MAC_BUF(var) char var[18] __attribute__((unused))
extern char *print_mac(char *buf, const u8 *addr);

/**
 * queue_delayed_work in pre 2.6.20 can't rearm from inside
 * the work member. So instead do a rather hacky sleep
 */
#define delayed_work work_struct
#define INIT_DELAYED_WORK INIT_WORK

static int inline efx_queue_delayed_work(struct workqueue_struct *wq,
					 struct work_struct *work,
					 unsigned long delay)
{
	if (unlikely(delay > 0))
		schedule_timeout_uninterruptible(delay);
	return queue_work(wq, work);
}
#define queue_delayed_work efx_queue_delayed_work

/**
 * The old and new work-function prototypes just differ
 * in the type of the pointer returned, so it's safe
 * to cast between the prototypes.
 */
typedef void (*efx_old_work_func_t)(void *p);

#undef INIT_WORK
#define INIT_WORK(_work, _func)					\
	do {							\
		INIT_LIST_HEAD(&(_work)->entry);		\
		(_work)->pending = 0;				\
		PREPARE_WORK((_work),				\
			     (efx_old_work_func_t) (_func),	\
			     (_work));				\
	} while (0)

#define napi_str napi_dev[0]

static inline void netif_napi_add(struct net_device *dev,
				  struct net_device *dummy,
				  int (*poll) (struct net_device *, int *),
				  int weight)
{
	dev->weight = weight;
	dev->poll = poll;
}

#define napi_enable netif_poll_enable
#define napi_disable netif_poll_disable

#define netif_rx_complete(dev, dummy) netif_rx_complete(dev)

#endif /* EFX_KERNEL_COMPAT_H */
